package com.theme.starterkit.libs;

import com.theme.starterkit.libs.config.KTThemeBaseConfig;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.*;

public class KTTheme {
    @Autowired
    private KTThemeBaseConfig settings;

    // Theme variables
    private boolean _modeSwitchEnabled = false;

    private String _modeDefault = "light";

    private String _direction = "ltr";

    private String _currentLayout = null;

    private final Map<String, Map<String, String>> _htmlAttributes = new HashMap<String, Map<String, String>>();

    private final Map<String, String[]> _htmlClasses = new HashMap<String, String[]>();

    // Keep page level assets
    private final List<String> _javascriptFiles = new ArrayList<String>();

    private final List<String> _cssFiles = new ArrayList<String>();

    private final List<String> _vendorFiles = new ArrayList<String>();

    @PostConstruct
    public void init(){
        KTBootstrap bootstrap = new KTBootstrap(this, settings);
        bootstrap.init();
        switch (this.getLayout()) {
            case "auth" -> bootstrap.initAuthLayout();
            case "default-dark-header" -> bootstrap.initDarkHeaderLayout();
            case "default-light-header" -> bootstrap.initLightHeaderLayout();
            case "default-dark-sidebar" -> bootstrap.initDarkSidebarLayout();
            case "default-light-sidebar" -> bootstrap.initLightSidebarLayout();
        }
    }

    // Add HTML attributes by scope
    public void addHtmlAttribute(String scope, String attributeName, String attributeValue)
    {
        Map<String, String> attributes;
        if (_htmlAttributes.containsKey(scope))
        {
            attributes = _htmlAttributes.get(scope);
        }
        else
        {
            attributes = new HashMap<String, String>();
        }

        attributes.put(attributeName, attributeValue);
        _htmlAttributes.put(scope, attributes);
    }

    // Add HTML class by scope
    public void addHtmlClass(String scope, String className)
    {
        List<String> list;
        if (_htmlClasses.containsKey(scope))
        {
            list = new ArrayList<String>(Arrays.asList(_htmlClasses.get(scope)));
        } else {
            list = new ArrayList<String>();
        }
        list.add(className);
        String[] array = new String[list.size()];
        _htmlClasses.put(scope, list.toArray(array));
    }

    // Print HTML attributes for the HTML template
    public String printHtmlAttributes(String scope)
    {
        List<String> list = new ArrayList<String>();
        if (_htmlAttributes.containsKey(scope))
        {
            _htmlAttributes.get(scope).forEach((key, value) -> {
                String item = key + "=" + value;
                list.add(item);
            });
            return String.join(",", list);
        }
        return "data-kt-no-attribute='true'";
    }

    // Print HTML classes for the HTML template
    public String printHtmlClasses(String scope)
    {
        if (_htmlClasses.containsKey(scope))
        {
            return String.join(" ", _htmlClasses.get(scope));
        }
        return "";
    }

    // Get SVG icon content
    public String getSvgIcon(String path, String classNames)
    {
        List<String> svgLines = new ArrayList<String>();
        try {
            File myObj = new File("./src/main/resources/static/assets/media/icons/" + path);
            Scanner myReader = new Scanner(myObj);
            while (myReader.hasNextLine()) {
                String line = myReader.nextLine();
                svgLines.add(line);
            }
            myReader.close();
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }

        StringBuilder output = new StringBuilder();

        output.append("<span class=\"").append(classNames).append("\">");
        for (String line : svgLines) {
            output.append(line);
        }
        output.append("</span>");

        return output.toString();
    }

    // Set dark mode enabled status
    public void setModeSwitch(boolean flag)
    {
        _modeSwitchEnabled = flag;
    }

    // Check dark mode status
    public boolean isModeSwitchEnabled()
    {
        return _modeSwitchEnabled;
    }

    // Set the mode to dark or light
    public void setModeDefault(String mode)
    {
        _modeDefault = mode;
    }

    // Get current mode
    public String getModeDefault()
    {
        return _modeDefault;
    }

    // Set style direction
    public void setDirection(String direction)
    {
        _direction = direction;
    }

    // Get style direction
    public String getDirection()
    {
        return _direction;
    }

    // Checks if style direction is RTL
    public boolean isRtlDirection()
    {
        return _direction.equalsIgnoreCase("rtl");
    }

    public String getAssetPath(String path)
    {
        return settings.getAssetsDir() + path;
    }

    public String getView(String path)
    {
        return settings.getLayoutDir() + path;
    }

    public String getPageView(String folder, String file)
    {
        return "pages/"+ folder +"/" + file;
    }

    // Extend CSS file name with RTL
    public String extendCssFilename(String path)
    {
        if (isRtlDirection())
        {
            path = path.replace(".css", ".rtl.css");
        }

        return path;
    }

    // Include favicon from settings
    public String getFavicon()
    {
        return getAssetPath(settings.getAssets().favicon);
    }

    // Include the fonts from settings
    public String[] getFonts()
    {
        String[] array = new String[settings.getAssets().fonts.size()];
        return settings.getAssets().fonts.toArray(array);
    }

    // Get the global assets
    public String[] getGlobalAssets(String type)
    {
        List<String> files =
                Objects.equals(type, "Css") ? settings.getAssets().css : settings.getAssets().js;
        List<String> newList = new ArrayList<String>();

        files.forEach((file) -> {
            if (Objects.equals(type, "Css"))
            {
                newList.add(getAssetPath(extendCssFilename(file)));
            }
            else
            {
                newList.add(getAssetPath(file));
            }
        });

        String[] array = new String[newList.size()];
        return newList.toArray(array);
    }

    // Add multiple vendors to the page by name
    public void addVendors(String[] vendors)
    {
        for(String vendor : vendors)
        {
            if (!_vendorFiles.contains(vendor))
            {
                _vendorFiles.add(vendor);
            }
        }
    }

    // Add single vendor to the page by name
    public void addVendor(String vendor)
    {
        if (!_vendorFiles.contains(vendor))
        {
            _vendorFiles.add(vendor);
        }
    }

    // Add custom javascript file to the page
    public void addJavascriptFile(String file)
    {
        if (!_javascriptFiles.contains(file))
        {
            _javascriptFiles.add(file);
        }
    }

    // Add custom CSS file to the page
    public void addCssFile(String file)
    {
        if (!_cssFiles.contains(file))
        {
            _cssFiles.add(file);
        }
    }

    public String[] getJavascriptFiles()
    {
        String[] array = new String[_javascriptFiles.size()];
        return _javascriptFiles.toArray(array);
    }

    public String[] getCssFiles()
    {
        String[] array = new String[_cssFiles.size()];
        return _cssFiles.toArray(array);
    }

    // Get vendor files from settings
    public String[] getVendors(String type)
    {
        Map<String, Map<String, String[]>> vendors = settings.getVendors();
        List<String> files = new ArrayList<String>();
        _vendorFiles.forEach(vendor -> {
            if (vendors.containsKey(vendor) && vendors.get(vendor).containsKey(type))
            {
                String[] vendorFiles = vendors.get(vendor).get(type);
                for(String file : vendorFiles)
                {
                    String vendorPath = file.contains("https://") ? file : getAssetPath(file);
                    files.add(vendorPath);
                }
            }
        });
        String[] array = new String[files.size()];
        return files.toArray(array);
    }

    public String getAttributeValue(String scope, String attributeName){
        if (_htmlAttributes.containsKey(scope))
        {
            if (_htmlAttributes.get(scope).containsKey(attributeName))
            {
                return _htmlAttributes.get(scope).get(attributeName);
            }
            return "";
        }

        return "";
    }

    public void setLayout(String layout){
        this._currentLayout = layout;
    }

    public String getLayout(){
        return this._currentLayout;
    }
}
